# -*- coding: utf-8 -*-
# Part of Cnmx. See LICENSE file for full copyright and licensing details.
import base64
import qrcode
import logging
from io import BytesIO

from odoo import api, fields, models, _
from odoo.exceptions import AccessError, UserError

_logger = logging.getLogger(__name__)


class VerifyCNTaxInvoice(models.TransientModel):
    _name = 'verify.cn.tax.invoice'
    _description = 'Verify China Tax Invoice'

    user_id = fields.Many2one("res.users")
    verify_type = fields.Selection([
        ("sms", "SMS"),
        ("qr", "QR"),
    ], required=True)
    sms_code = fields.Char()
    qr_code_image = fields.Binary("QR Code")

    @api.model
    def make_qr_code_image(self, code):
        qr = qrcode.QRCode(
            version=1,
            error_correction=qrcode.constants.ERROR_CORRECT_L,
            box_size=10,
            border=4,
        )
        qr.add_data(code)
        qr.make(fit=True)

        img = qr.make_image(fill_color="black", back_color="white")
        buffer = BytesIO()
        img.save(buffer, format="PNG")
        qr_base64 = base64.b64encode(buffer.getvalue())
        return qr_base64

    def verify_code(self):
        # 验证短信 验证成功后进行是否需要人脸识别接口的查询
        sms_code = self.sms_code and self.sms_code.strip()
        if not sms_code:
            # 请输入验证码
            raise UserError("Please enter the verification code.")
        try:
            result = self.env.company.login_cn_tax(self.user_id, sms_code)
            if result.get("code", 0) != 200:
                raise UserError(result.get("msg", ""))
        except Exception as error:
            _logger.error("wizard verify code error: %s", error)
            raise UserError(str(error))

        # 验证成功后进行是否需要人脸识别接口的查询
        try:
            face_state = self.env.company.get_cn_tax_face_state(self.user_id)
            if face_state is False:
                self.user_id.property_cn_state = "logged_in"
            else:
                self.user_id.property_cn_state = "pending_face_recognition"
        except Exception as error:
            _logger.error("wizard get face state error: %s", error)
            raise UserError(str(error))

    def check_face_state(self):
        # 人脸识别结束后重新查询一次状态
        try:
            face_state = self.env.company.get_cn_tax_face_state(self.user_id)
            if face_state is False:
                self.user_id.property_cn_state = "logged_in"
        except Exception as error:
            _logger.error("wizard check face state error: %s", error)
            raise UserError(str(error))

