# -*- coding: utf-8 -*-
# Part of MX. See LICENSE file for full copyright and licensing details.
import logging

from odoo import api, fields, models, _
from odoo.exceptions import AccessError, ValidationError, UserError

_logger = logging.getLogger(__name__)


class ResUsers(models.Model):
    _inherit = 'res.users'

    cn_username = fields.Char(copy=False)
    cn_password = fields.Char(copy=False)
    # 01-法定代表人 02-财务负责人 03-办税员 04-涉税服务人员 05-管理员 07-领票人 08-社保经办人 09-开票员 10-销售人员
    property_cn_identity = fields.Selection([
        ('01', 'Legal Representative'),
        ('02', 'Financial Officer'),
        ('03', 'Tax Clerk'),
        ('04', 'Tax Service Provider'),
        ('05', 'Administrator'),
        ('07', 'Invoice Receiver'),
        ('08', 'Social Security Agent'),
        ('09', 'Invoice Issuer'),
        ('10', 'Salesperson'),
    ], copy=False, company_dependent=True)
    # 已登录  待人脸识别  未登录
    property_cn_state = fields.Selection([
        ('logged_in', 'Logged In'),
        ('pending_face_recognition', 'Pending Face Recognition'),
        ('not_logged_in', 'Not Logged In')
    ], string='Login State', copy=False, company_dependent=True, default="not_logged_in")

    @property
    def SELF_READABLE_FIELDS(self):
        return super().SELF_WRITEABLE_FIELDS + ['cn_username', 'cn_password', 'property_cn_identity', 'property_cn_state']

    @property
    def SELF_WRITEABLE_FIELDS(self):
        return super().SELF_WRITEABLE_FIELDS + ['cn_username', 'cn_password', 'property_cn_identity', 'property_cn_state']

    def log_out_cn_tax(self):
        """将用户状态设置为 未登录"""
        self.sudo().write({
            "property_cn_state": "not_logged_in"
        })

    def login_cn_tax(self):
        """登录"""
        self.ensure_one()
        try:
            result = self.env.company.login_cn_tax(self)
            code = result.get("code", 0)
            if code != 200:
                error_msg = result.get("msg", "") or result.get("message", "")
                raise AccessError(error_msg)
        except Exception as error:
            _logger.error("login cn tax error: %s", error)
            raise UserError(str(error))
        wizard = self.env["verify.cn.tax.invoice"].create({
            "user_id": self.id,
            "verify_type": "sms",
        })
        action = self.env["ir.actions.actions"]._for_xml_id("l10n_cn_edi.action_verify_cn_tax_invoice")
        action["res_id"] = wizard.id
        return action

    def verify_face(self):
        """  人脸识别
           当登录状态为`待人脸识别`时,需要先进行登录状态的检查,如果是登录的情况才继续调用返回人脸识别二维码
        """
        self.ensure_one()
        try:
            result = self.env.company.get_cn_tax_face_img(self)
            code = result.get("code", 0)
            if code == 302:
                self.log_out_cn_tax()
                error_msg = result.get("msg", "")
                return {
                    'type': 'ir.actions.client',
                    'tag': 'display_notification',
                    'params': {
                        'title': _("Verification failed"),
                        'message': error_msg,
                        'next': {'type': 'ir.actions.act_window_close'},
                        'type': 'danger',
                        'sticky': True,
                    }
                }
            elif code != 200:
                error_msg = result.get("msg", "")
                raise AccessError(error_msg)
            qr_code = result.get("data", {}).get("ewm", "")
            qr_code_image = self.env["verify.cn.tax.invoice"].make_qr_code_image(qr_code)
            wizard = self.env["verify.cn.tax.invoice"].create({
                "user_id": self.id,
                "verify_type": "qr",
                "qr_code_image": qr_code_image
            })
            action = self.env["ir.actions.actions"]._for_xml_id("l10n_cn_edi.action_verify_cn_tax_invoice")
            action["res_id"] = wizard.id
            return action
        except Exception as error:
            _logger.error("verify face error: %s", error)
            raise UserError(str(error))


