# -*- coding: utf-8 -*-
# Part of MX. See LICENSE file for full copyright and licensing details.
import json
import base64
import logging

from odoo import api, fields, models, _
from odoo.exceptions import UserError, ValidationError
from odoo.addons.l10n_cn_edi.tools.tax_invoice_api import ForticClient


_logger = logging.getLogger(__name__)


class ResCompany(models.Model):
    _inherit = 'res.company'

    # 开票人
    cn_post_user_id = fields.Many2one("res.users", copy=False)

    def log_out_cn_tax(self):
        """将公司的税控开票人状态设置为 未登录"""
        self.cn_post_user_id.log_out_cn_tax()

    def check_cn_tax_login_info(self, post_user_id):
        self.ensure_one()
        if not self.vat:
            # 请先设置公司税号
            raise ValidationError(_("Please set the company tax identification number first."))
        if not post_user_id:
            # 请先设置公司的开票人
            raise ValidationError(_("Please set the company's invoicing operator first."))
        if not post_user_id.cn_username or not post_user_id.cn_password:
            # 请检查开票人的账号密码是否已设置
            raise ValidationError(
                _("Please check whether the invoicing operator's username and password are configured."))

    def login_cn_tax(self, post_user_id, sms_code=""):
        post_user_id = post_user_id or self.cn_post_user_id
        # 检查税号/开票人/开票人信息
        self.check_cn_tax_login_info(post_user_id)
        token = self.env['ir.config_parameter'].sudo().get_param('l10n_cn_edi.fortic_api_keys')
        client = ForticClient(token)
        result = client.login(self.vat, post_user_id.cn_username, post_user_id.cn_password,
                              post_user_id.property_cn_identity, sms_code)
        return result

    # 登录 + 验证码校验
    # 检查是否需要人脸识别 & 同时用此接口检查登录状态
    def check_cn_tax_face_state(self, post_user_id):
        post_user_id = post_user_id or self.cn_post_user_id
        token = self.env['ir.config_parameter'].sudo().get_param('l10n_cn_edi.fortic_api_keys')
        client = ForticClient(token)
        result = client.check_face_state(self.vat, post_user_id.cn_username)
        return result

    # 检查是否需要人脸识别 & 同时用此接口检查登录状态
    def get_cn_tax_face_state(self, post_user_id):
        result = self.check_cn_tax_face_state(post_user_id)
        if result.get("code", 0) == 200:
            if result.get("data", ""):
                s = result.get("data")
                # 补齐缺失的 =
                missing_padding = len(s) % 4
                if missing_padding:
                    s += "=" * (4 - missing_padding)

                decoded = json.loads(base64.b64decode(s).decode("utf-8"))
                if decoded.get("Sfsl", "") == "N":
                    return False
                else:
                    return True
            else:
                return result.get("msg", "")
        else:
            return result.get("msg", "")

    # 人脸识别
    def get_cn_tax_face_img(self, post_user_id):
        post_user_id = post_user_id or self.cn_post_user_id
        token = self.env['ir.config_parameter'].sudo().get_param('l10n_cn_edi.fortic_api_keys')
        client = ForticClient(token)
        result = client.get_face_img(self.vat, post_user_id.cn_username)
        return result

    # 蓝字开票
    def post_cn_tax_invoice(self, invoice_data, post_user_id=None):
        post_user_id = post_user_id or self.cn_post_user_id
        if not post_user_id or post_user_id.property_cn_state != "logged_in":
            # 用户%s未在税控系统登录
            raise UserError(_("The user %s has not logged in to the tax control system", post_user_id.name))
        token = self.env['ir.config_parameter'].sudo().get_param('l10n_cn_edi.fortic_api_keys')
        client = ForticClient(token)
        result = client.post_invoice(self.vat, post_user_id.cn_username, invoice_data)
        return result

    # 获取蓝字开票文件
    def get_cn_tax_invoice_pdf(self, invoice_number, post_user_id=None):
        post_user_id = post_user_id or self.cn_post_user_id
        token = self.env['ir.config_parameter'].sudo().get_param('l10n_cn_edi.fortic_api_keys')
        client = ForticClient(token)
        result = client.get_invoice_pdf(self.vat, post_user_id.cn_username, invoice_number)
        return result

    # 红字开票
    def post_cn_tax_invoice_refund(self, invoice_data, post_user_id=None):
        post_user_id = post_user_id or self.cn_post_user_id
        if post_user_id.property_cn_state != "logged_in":
            # 用户%s未在税控系统登录
            raise UserError(_("The user %s has not logged in to the tax control system", post_user_id.name))
        token = self.env['ir.config_parameter'].sudo().get_param('l10n_cn_edi.fortic_api_keys')
        client = ForticClient(token)
        result = client.post_refund_invoice(self.vat, post_user_id.cn_username, invoice_data)
        return result

    # 获取红字开票文件
    def get_cn_tax_invoice_refund_pdf(self, refund_invoice_number, post_user_id=None):
        post_user_id = post_user_id or self.cn_post_user_id
        token = self.env['ir.config_parameter'].sudo().get_param('l10n_cn_edi.fortic_api_keys')
        client = ForticClient(token)
        result = client.get_refund_invoice_pdf(self.vat, post_user_id.cn_username, refund_invoice_number)
        return result

    def _cron_check_user_cn_tax_state(self):
        """检查用户的登录状态并更新"""
        companies = self.env["res.company"].sudo().search([('cn_post_user_id', '!=', False)])
        for company in companies.sudo():
            try:
                if company.cn_post_user_id.with_company(company).property_cn_state == "not_logged_in":
                    continue
                face_state = company.get_cn_tax_face_state(company.cn_post_user_id)
                if face_state is False:
                    company.cn_post_user_id.with_company(company).property_cn_state = "logged_in"
                elif isinstance(face_state, str):
                    company.cn_post_user_id.with_company(company).property_cn_state = "not_logged_in"
                else:
                    company.cn_post_user_id.with_company(company).property_cn_state = "pending_face_recognition"
            except Exception as error:
                _logger.error("cron check user cn tax state error: %s", str(error))
