from odoo import models, fields, api
from odoo.osv import expression
from odoo.exceptions import ValidationError


class ResPartner(models.Model):
    _inherit = "res.partner"

    invoice_header_ids = fields.One2many('invoice.header', 'partner_id')


class AccountMove(models.Model):
    _inherit = "account.move"

    invoice_header_id = fields.Many2one('invoice.header', copy=True)

    @api.onchange('partner_id')
    def _onchange_partner_id_with_invoice_header(self):
        self.invoice_header_id = self.partner_id.invoice_header_ids[-1:]


class InvoiceHeader(models.Model):
    _name = "invoice.header"
    _description = "Invoice Header"

    name = fields.Char(string='Name', required=True)
    vat = fields.Char(size=18, required=True)
    mobile = fields.Char(unaccent=False)
    street = fields.Char()
    street2 = fields.Char()
    zip = fields.Char(change_default=True)
    city = fields.Char()
    state_id = fields.Many2one("res.country.state", string='State', ondelete='restrict',
                               domain="[('country_id', '=?', country_id)]")
    country_id = fields.Many2one('res.country', string='Country', ondelete='restrict')
    country_code = fields.Char(related='country_id.code', string="Country Code")
    invoice_type = fields.Selection(selection=[
        ("special", "VAT Special Invoice"),
        ("normal", "VAT Ordinary Invoice"),
    ], string="Invoice Type", default="normal", required=True)
    bank_id = fields.Many2one('res.partner.bank', string='Bank')
    partner_id = fields.Many2one('res.partner')

    def create(self, vals):
        res = super(InvoiceHeader, self).create(vals)
        # 企业只能绑定一个抬头
        companies = res.partner_id.filtered(lambda x: x.company_type == 'company' and len(x.invoice_header_ids) > 1)
        if companies:
            raise ValidationError(
                "Company %s has multiple associated invoice headers." % ('.'.join(companies.mapped("name"))))
        return res

    def write(self, vals):
        res = super(InvoiceHeader, self).write(vals)
        # 企业只能绑定一个抬头
        companies = self.partner_id.filtered(lambda x: x.company_type == 'company' and len(x.invoice_header_ids) > 1)
        if companies:
            raise ValidationError(
                "Company %s has multiple associated invoice headers." % ('.'.join(companies.mapped("name"))))
        return res
