# -*- coding: utf-8 -*-
# Part of MX. See LICENSE file for full copyright and licensing details.
import json
import base64
import logging
from collections import defaultdict


from odoo import fields, models, _
from odoo.exceptions import UserError

_logger = logging.getLogger(__name__)


class AccountEdiFormat(models.Model):
    _inherit = 'account.edi.format'

    # -------------------------------------------------------------------------
    # CN EDI
    # -------------------------------------------------------------------------

    def _l10n_cn_tax_invoice_web_service_sign(self, invoices, info_list):
        invoices_results = {}
        for index, invoice in enumerate(invoices):
            error_msg = ""
            try:
                # 没有发票号 先获取发票号
                if not invoice.fapiao:
                    invoice_data = info_list[index]
                    result = invoice.company_id.post_cn_tax_invoice(invoice_data)
                    if result.get("code", 0) == 200:
                        data = result.get("data")
                        if not data or not isinstance(data, dict):
                            error_msg = result.get("msg", "")
                            raise UserError(error_msg)

                        if data.get("Fphm"):
                            invoice_number = data.get("Fphm")
                            invoice.fapiao = invoice_number
                            invoices_results[invoice] = {'success': True}
                        # 需要人脸识别
                        elif data.get("ewm"):
                            invoice.company_id.cn_post_user_id.with_company(
                                invoice.company_id).property_cn_state = "pending_face_recognition"
                            # 请先进行人脸识别
                            error_msg = _("Please perform facial recognition first.")
                        else:
                            error_msg = result.get("msg", "")
                    elif result.get("code", 0) == 302 and result.get("msg", "") == "请您获取验证码重新登录":
                        invoice.company_id.log_out_cn_tax()
                        error_msg = "请您获取验证码重新登录"
                    else:
                        error_msg = result.get("msg", "")
                # 有发票号 不管是刚获取的还是历史的都去获取发票的pdf
                if invoice.fapiao:
                    invoice_number = invoice.fapiao
                    pdf_result = invoice.company_id.get_cn_tax_invoice_pdf(invoice_number)
                    if pdf_result.get("code", 0) == 200 and pdf_result.get("data", ""):
                        s = pdf_result.get("data")
                        # 补齐缺失的 =
                        missing_padding = len(s) % 4
                        if missing_padding:
                            s += "=" * (4 - missing_padding)

                        pdf_bytes = base64.b64decode(s)
                        # 创建 ir.attachment 记录
                        attachment = self.env['ir.attachment'].create({
                            'name': "invoice-%s.pdf" % invoice_number,
                            'type': 'binary',
                            'datas': base64.b64encode(pdf_bytes),  # 注意这里再 base64 一次，存数据库
                            'res_model': 'account.move',  # 关联模型
                            'res_id': invoice.id,  # 关联发票
                            'mimetype': 'application/pdf',
                        })
                        invoices_results[invoice] = {'success': True, 'attachment': attachment}
                        invoice.message_post(
                            body=_(
                                "The invoice has been successfully issued at the tax bureau (invoice number : %s)",
                                invoice_number), attachment_ids=[attachment.id])
                    elif pdf_result.get("code", 0) == 302 and pdf_result.get("msg", "") == "请您获取验证码重新登录":
                        invoice.company_id.log_out_cn_tax()
                        error_msg = "请您获取验证码重新登录"
                    else:
                        if invoices_results.get(invoice, {}).get("success"):
                            invoice.message_post(
                                body=_(
                                    "The invoice has been successfully issued at the tax bureau (invoice number : %s)",
                                    invoice_number))
                        error_msg = _("Get invoice pdf error: %s", pdf_result.get("msg", ""))
            except Exception as error:
                _logger.error("cn tax invoice error: %s", error)
                error_msg = str(error)

            if error_msg:
                invoices_results[invoice] = {
                    'error': error_msg,
                    'blocking_level': 'warning',
                }

        return invoices_results

    def _l10n_cn_tax_invoice_refund_web_service_sign(self, invoices, info_list):
        invoices_results = {}
        for index, invoice in enumerate(invoices):
            error_msg = ""
            try:
                # 没有红字发票号 先获取红字发票号
                if not invoice.fapiao:
                    invoice_data = info_list[index]
                    result = invoice.company_id.post_cn_tax_invoice_refund(invoice_data)
                    if result.get("code", 0) == 200:
                        data = result.get("data")
                        if not data or not isinstance(data, dict):
                            error_msg = result.get("msg", "")
                            raise UserError(error_msg)

                        if data.get("xxbbh"):
                            refund_invoice_number = data.get("xxbbh")
                            invoice.fapiao = refund_invoice_number
                            invoices_results[invoice] = {'success': True}
                        # 需要人脸识别
                        elif data.get("ewm"):
                            invoice.company_id.cn_post_user_id.with_company(
                                invoice.company_id).property_cn_state = "pending_face_recognition"
                            # 请先进行人脸识别
                            error_msg = _("Please perform facial recognition first.")
                        else:
                            error_msg = result.get("msg", "")
                    elif result.get("code", 0) == 302 and result.get("msg", "") == "请您获取验证码重新登录":
                        invoice.company_id.log_out_cn_tax()
                        error_msg = "请您获取验证码重新登录"
                    else:
                        error_msg = result.get("msg", "")
                # 有发票号 不管是刚获取的还是历史的都去获取发票的pdf
                if invoice.fapiao:
                    refund_invoice_number = invoice.fapiao
                    pdf_result = invoice.company_id.get_cn_tax_invoice_refund_pdf(refund_invoice_number)
                    if pdf_result.get("code", 0) == 200 and pdf_result.get("data", ""):
                        s = pdf_result.get("data")
                        # 补齐缺失的 =
                        missing_padding = len(s) % 4
                        if missing_padding:
                            s += "=" * (4 - missing_padding)

                        pdf_bytes = base64.b64decode(s)
                        # 创建 ir.attachment 记录
                        attachment = self.env['ir.attachment'].create({
                            'name': "refund_invoice-%s.pdf" % refund_invoice_number,
                            'type': 'binary',
                            'datas': base64.b64encode(pdf_bytes),  # 注意这里再 base64 一次，存数据库
                            'res_model': 'account.move',  # 关联模型
                            'res_id': invoice.id,  # 关联发票
                            'mimetype': 'application/pdf',
                        })
                        invoices_results[invoice] = {'success': True, 'attachment': attachment}
                        invoice.message_post(
                            body=_(
                                "The refund invoice has been successfully issued at the tax bureau (refund invoice number : %s)",
                                refund_invoice_number), attachment_ids=[attachment.id])
                    elif pdf_result.get("code", 0) == 302 and pdf_result.get("msg", "") == "请您获取验证码重新登录":
                        invoice.company_id.log_out_cn_tax()
                        error_msg = "请您获取验证码重新登录"
                    else:
                        if invoices_results.get(invoice, {}).get("success"):
                            invoice.message_post(body=_(
                                "The refund invoice has been successfully issued at the tax bureau (refund invoice number : %s)",
                                refund_invoice_number))
                        error_msg = _("Get refund invoice pdf error: %s", pdf_result.get("msg", ""))
            except Exception as error:
                _logger.error("cn tax refund invoice error: %s", error)
                error_msg = str(error)

            if error_msg:
                invoices_results[invoice] = {
                    'error': error_msg,
                    'blocking_level': 'warning',
                }

        return invoices_results

    def _l10n_cn_tax_get_invoices_info(self, invoices):
        info_list = []
        wrong_result = {}
        for invoice in invoices:
            # 专票 or 普票
            invoice_type = invoice.invoice_header_id.invoice_type or ""
            # 基础信息
            info = {
                "fplxdm": "81" if invoice_type == "special" else "82",
                "kplx": "0",
                "zsfs": "0",
            }

            # 销方信息
            sale_partner = invoice.company_id.partner_id
            info.update({
                "xhdwsbh": sale_partner.vat,
                "xhdwmc": sale_partner.name,
                "xhdwdzdh": "%s%s%s%s %s".strip() % (
                    sale_partner.state_id.name or "", sale_partner.city or "", sale_partner.street or "",
                    sale_partner.street2 or "", sale_partner.phone or sale_partner.mobile or ""),
                "xhdwyhzh": "%s %s".strip() % (
                    sale_partner.bank_ids[:1].bank_id.name or "", sale_partner.bank_ids[:1].acc_number or ""),
            })

            # 购方信息
            if invoice.invoice_header_id:
                purchase_partner = invoice.invoice_header_id
            else:
                purchase_partner = invoice.partner_id
            info["ghdwmc"] = purchase_partner.name
            if purchase_partner.vat:
                info["ghdwsbh"] = purchase_partner.vat

            # 明细行信息
            cn_currency = self.env.ref("base.CNY")
            need_convert = False
            if invoice.currency_id != cn_currency:
                need_convert = True

            index = 0
            amount = 0
            no_tax_amount = 0
            tax_amount = 0
            invoice_lines = invoice.invoice_line_ids.filtered(
                lambda line: line.display_type not in ('line_note', 'line_section'))
            # 折扣行
            discount_invoice_lines = invoice_lines.filtered(lambda l: l.product_id.is_cn_tax_discount)
            # 正常发票行
            normal_invoice_lines = invoice_lines - discount_invoice_lines
            # 被折扣发票行
            matched_to_discount_lines = self.env['account.move.line']
            # 折扣行和被折扣行的对照 key=折扣行 value=被折扣行
            discount_line_dict = {}
            # 折扣行和被折扣行的对照 key=折扣行 value=被折扣行
            to_discount_line_dict = {}
            # 如果发生折扣行无法匹配到可用的折扣行 break当前发票的税控开票info生成
            break_flag = False
            for discount_line in discount_invoice_lines.sorted('price_total'):
                to_discount_lines = (normal_invoice_lines - matched_to_discount_lines).sorted('price_total').filtered(
                    lambda x: abs(x.price_total) >= abs(discount_line.price_total))
                if to_discount_lines:
                    to_discount_line = to_discount_lines[0]
                    discount_line_dict[discount_line] = to_discount_line
                    to_discount_line_dict[to_discount_line] = discount_line
                    matched_to_discount_lines |= to_discount_line
                else:
                    # 折扣行%s无法匹配到可用的被折扣行
                    message = _("The discount line %s could not be matched with an available discounted line.") % discount_line.product_id.display_name
                    wrong_result[invoice] = {
                        'error': message,
                        'blocking_level': 'error'}
                    break_flag = True
                    break
            if break_flag:
                break

            # 因为折扣行下面必须跟着被折扣行 所以需要把遍历的发票行重新排序
            to_do_lines = []
            for line in normal_invoice_lines:
                to_do_lines.append(line)
                if to_discount_line_dict.get(line):
                    to_do_lines.append(to_discount_line_dict.get(line))

            for line in to_do_lines:
                if not line._need_post_cn_tax_invoice():
                    continue
                line.cn_tax_line_number = index + 1
                # 金额换算成人民币
                if need_convert:
                    price_unit = invoice.currency_id._convert(
                        from_amount=line.price_unit,
                        to_currency=cn_currency,
                        company=invoice.company_id,
                        date=invoice.invoice_date or fields.Date.today(),
                    )
                    price_subtotal = invoice.currency_id._convert(
                        from_amount=line.price_subtotal,
                        to_currency=cn_currency,
                        company=invoice.company_id,
                        date=invoice.invoice_date or fields.Date.today(),
                    )
                    price_total = invoice.currency_id._convert(
                        from_amount=line.price_total,
                        to_currency=cn_currency,
                        company=invoice.company_id,
                        date=invoice.invoice_date or fields.Date.today(),
                    )
                else:
                    price_unit = line.price_unit
                    price_subtotal = line.price_subtotal
                    price_total = line.price_total
                price_total = round(price_total, 2)
                price_subtotal = round(price_subtotal, 2)
                price_tax = round(cn_currency.round(price_total - price_subtotal), 2)

                # 累计金额
                amount += price_total
                no_tax_amount += price_subtotal
                tax_amount += price_tax

                # 税局商品编码 + 发票行性质
                if line.product_id.is_cn_tax_discount:
                    # 被折扣行
                    discount_line = discount_line_dict.get(line)
                    # 发票行性质-折扣行
                    fphxz = "1"
                    cn_tax_product_id = discount_line.product_id.with_company(invoice.company_id)._get_cn_tax_product_id()
                else:
                    # 发票行性质-正常行 -> 0  发票行性质-被折扣行 -> 2
                    fphxz = "2" if line in matched_to_discount_lines else "0"
                    cn_tax_product_id = line.product_id.with_company(invoice.company_id)._get_cn_tax_product_id()

                # 含税还是不含税
                hsbz = 1 if line.tax_ids[0].price_include else 0
                # 金额字段取值： 含税 -> 合计  不含税 -> 小计
                je = price_total if hsbz else price_subtotal
                if fphxz == "1":
                    info.update({
                        f"fyxm[{index}][fphxz]": fphxz,
                        f"fyxm[{index}][spmc]": f"*{cn_tax_product_id.name}*{line.product_id.name}*",
                        f"fyxm[{index}][je]": f"{je}",
                        f"fyxm[{index}][sl]": f"{line.tax_ids[0].amount / 100 or 0}",
                        f"fyxm[{index}][se]": f"{price_tax}",
                        f"fyxm[{index}][hsbz]": f"{hsbz}",
                        f"fyxm[{index}][spbm]": f"{cn_tax_product_id.code}",
                    })
                else:
                    info.update({
                        f"fyxm[{index}][fphxz]": fphxz,
                        f"fyxm[{index}][spmc]": f"*{cn_tax_product_id.name}*{line.product_id.name}*",
                        # f"fyxm[{index}][ggxh]": line.name or "",
                        f"fyxm[{index}][dw]": line.product_uom_id.name or "",
                        f"fyxm[{index}][spsl]": f"{line.quantity}",
                        f"fyxm[{index}][dj]": f"{price_unit}",
                        f"fyxm[{index}][je]": f"{je}",
                        f"fyxm[{index}][sl]": f"{line.tax_ids[0].amount / 100 or 0}",
                        f"fyxm[{index}][se]": f"{price_tax}",
                        f"fyxm[{index}][hsbz]": f"{hsbz}",
                        f"fyxm[{index}][spbm]": f"{cn_tax_product_id.code}",
                        # f"fyxm[{index}][yhzcbs]": "0",
                        # f"fyxm[{index}][lslbs]": "0",
                        # f"fyxm[{index}][zzstsgl]": "0",
                    })

                index += 1

            # 合计 + 基础信息
            info.update({
                "hjje": f"{round(no_tax_amount, 2)}",
                "hjse": f"{round(tax_amount, 2)}",
                "jshj": f"{round(amount, 2)}",
                # "kpr": f"{invoice.post_user.name}",
                # "fhr": f"{invoice.check_user.name}",
                "bz": (invoice.narration and invoice.narration[:200]) or ""
            })

            info_list.append(info)
        return info_list, wrong_result

    def _l10n_cn_tax_get_refund_invoices_info(self, invoices):
        info_list = []
        for invoice in invoices:
            # 专票 or 普票
            invoice_type = invoice.reversed_entry_id.invoice_header_id.invoice_type or invoice.invoice_header_id.invoice_type or ""
            # 基础信息
            info = {
                "fplxdm": "028" if invoice_type == "special" else "026",
                "chyydm": "01",
                "sqyy": "2"
            }

            # 发票信息
            info["yfphm"] = invoice.reversed_entry_id.fapiao

            # 销方信息
            sale_partner = invoice.company_id.partner_id
            info["xhdwsbh"] = sale_partner.vat

            # 明细行信息
            cn_currency = self.env.ref("base.CNY")
            need_convert = False
            if invoice.currency_id != cn_currency:
                need_convert = True

            index = 0
            amount = 0
            no_tax_amount = 0
            tax_amount = 0
            for line in invoice.invoice_line_ids.filtered(
                    lambda line: line.display_type not in ('line_note', 'line_section')):
                if not line.cn_tax_line_number:
                    continue

                # 金额换算成人民币
                if need_convert:
                    price_subtotal = invoice.currency_id._convert(
                        from_amount=line.price_subtotal,
                        to_currency=cn_currency,
                        company=invoice.company_id,
                        date=invoice.invoice_date or fields.Date.today(),
                    )
                    price_total = invoice.currency_id._convert(
                        from_amount=line.price_total,
                        to_currency=cn_currency,
                        company=invoice.company_id,
                        date=invoice.invoice_date or fields.Date.today(),
                    )
                else:
                    price_subtotal = line.price_subtotal
                    price_total = line.price_total

                price_total = round(price_total, 2)
                price_subtotal = round(price_subtotal, 2)
                price_tax = round(cn_currency.round(price_total - price_subtotal), 2)

                # 累计金额
                amount += price_total
                no_tax_amount += price_subtotal
                tax_amount += price_tax

                # 含税还是不含税
                hsbz = 1 if line.tax_ids[0].price_include else 0
                # 金额字段取值： 含税 -> 合计  不含税 -> 小计
                je = price_total if hsbz else price_subtotal

                info.update({
                    f"fyxm[{index}][xh]": f"{line.cn_tax_line_number}",
                    f"fyxm[{index}][spsl]": f"{line.quantity * -1}",
                    f"fyxm[{index}][je]": f"{je * -1}",
                    f"fyxm[{index}][se]": f"{price_tax * -1}",
                    f"fyxm[{index}][hsbz]": f"{hsbz}",
                })

                index += 1

            # 合计 + 基础信息
            info.update({
                "hjje": f"{round(no_tax_amount, 2) * -1}",
                "hjse": f"{round(tax_amount, 2) * -1}",
            })

            info_list.append(info)
        return info_list

    # -------------------------------------------------------------------------
    # EDI OVERRIDDEN METHODS
    # -------------------------------------------------------------------------

    def _l10n_cn_tax_invoice_content(self, invoice):
        return json.dumps(self._l10n_cn_tax_get_invoices_info(invoice)).encode()

    def _l10n_cn_tax_invoice_refund_content(self, invoice):
        return json.dumps(self._l10n_cn_tax_get_refund_invoices_info(invoice)).encode()

    def _get_move_applicability(self, move):
        """获取待执行的方法列表"""
        self.ensure_one()
        if self.code != 'cn_tax_invoice':
            return super()._get_move_applicability(move)

        if move.currency_id.name == "CNY" and move.move_type == "out_invoice":
            return {
                'post': self._l10n_cn_tax_invoice_post,
                # 'edi_content': self._l10n_cn_tax_invoice_content,
                # 'cancel': self._l10n_cn_tax_invoice_refund_post,
            }
        elif move.currency_id.name == "CNY" and move.move_type == "out_refund" and move.reversed_entry_id.fapiao:
            return {
                'post': self._l10n_cn_tax_invoice_refund_post,
                # 'edi_content': self._l10n_cn_tax_invoice_refund_content,
                # 'cancel': self._l10n_cn_tax_invoice_post,
            }

    def _needs_web_services(self):
        """返回True 为异步 在自动动作运行时操作  否则为同步 在过账的同时发生操作"""
        return self.code == 'cn_tax_invoice' or super()._needs_web_services()

    def _check_move_configuration(self, move):
        """过账生成edi document前检查是否有必须要用到的字段信息没有维护"""
        res = super()._check_move_configuration(move)
        if self.code != 'cn_tax_invoice':
            return res

        if not move.company_id.vat:
            res.append(_("VAT number is missing on company %s", move.company_id.display_name))
        if not move.company_id.partner_id.bank_ids:
            res.append(_("Bank number is missing on company %s", move.company_id.display_name))
        if not move.company_id.street:
            res.append(_("Address is missing on company %s", move.company_id.display_name))
        if not move.company_id.cn_post_user_id:
            res.append(_("Chinese tax invoice issuer is missing on company %s", move.company_id.display_name))
        for line in move.invoice_line_ids.filtered(lambda line: line.display_type not in ('line_note', 'line_section')):
            if not line._need_post_cn_tax_invoice():
                continue
            if line.product_id.is_cn_tax_discount:
                continue
            cn_tax_product_id = line.product_id.with_company(move.company_id)._get_cn_tax_product_id()
            if not cn_tax_product_id:
                # 请维护产品的税局商品编码
                res.append(
                    _("Please maintain the tax bureau commodity code of the product %s.", line.product_id.display_name))
        return res

    def _is_compatible_with_journal(self, journal):
        """判断此edi format是否在日记账上显示"""
        if self.code != 'cn_tax_invoice':
            return super()._is_compatible_with_journal(journal)
        return journal.country_code == 'CN' and journal.type == 'sale'

    def _l10n_cn_tax_invoice_post(self, invoices):
        """向税局发送开票请求"""
        wrong_res = {}
        no_vat_companies = invoices.company_id.filtered(lambda c: not c.vat)
        if no_vat_companies:
            wrong_invoices = invoices.filtered(lambda m: m.company_id in no_vat_companies)
            wrong_res = {inv: {
                'error': _("Please configure the company's tax number."),
                'blocking_level': 'error',
            } for inv in wrong_invoices}
            invoices -= wrong_invoices

        info_list, wrong_result = self._l10n_cn_tax_get_invoices_info(invoices)
        if wrong_result:
            wrong_res.update(wrong_result)
            for inv in wrong_result.keys():
                invoices -= inv

        res = self._l10n_cn_tax_invoice_web_service_sign(invoices, info_list)
        if wrong_res:
            res.update(wrong_res)
        return res

    def _l10n_cn_tax_invoice_refund_post(self, invoices):
        """向税局发送开票请求"""
        wrong_res = {}
        no_vat_companies = invoices.company_id.filtered(lambda c: not c.vat)
        if no_vat_companies:
            wrong_invoices = invoices.filtered(lambda m: m.company_id in no_vat_companies)
            wrong_res = {inv: {
                'error': _("Please configure the company's tax number."),
                'blocking_level': 'error',
            } for inv in wrong_invoices}
            invoices -= wrong_invoices

        info_list = self._l10n_cn_tax_get_refund_invoices_info(invoices)

        res = self._l10n_cn_tax_invoice_refund_web_service_sign(invoices, info_list)
        if wrong_res:
            res.update(wrong_res)
        return res
